/***************************************************************
 * Name:      eventmsgdialog.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2013
 * License:   GPL3
 **************************************************************/

#include "dxdefs.h"
#include "eventmsgdialog.h"
#include "eventdialog.h"
#include "dxsettings.h"
#include "data/engine.h"
#include "dxutils.h"
#include "dxicons.h"

IMPLEMENT_DYNAMIC_CLASS(LaterDialog, wxDialog)

BEGIN_EVENT_TABLE(LaterDialog, wxDialog)
    EVT_CHOICE(ID_REMINDERCHOICE, LaterDialog::OnReminder)
    EVT_BUTTON(ID_REMINDEREDIT, LaterDialog::OnReminderEdit)
END_EVENT_TABLE()

LaterDialog::LaterDialog(wxWindow* parent)
: wxDialog(parent, wxID_ANY, _("Reminder time")), m_reminder(5), m_reminderOwn(5)
{
    m_mainSizer = new wxBoxSizer(wxVERTICAL);

    wxFlexGridSizer *gridSizer = new wxFlexGridSizer(3, 5, 5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Reminder:")), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    wxArrayString choices;
    choices.Add(_("5 minutes after"));
    choices.Add(_("10 minutes after"));
    choices.Add(_("15 minutes after"));
    choices.Add(_("30 minutes after"));
    choices.Add(_("45 minutes after"));
    choices.Add(_("1 hour after"));
    choices.Add(_("2 hours after"));
    choices.Add(_("1 day after"));
    choices.Add(_("Custom"));
    m_reminderCtrl = new wxChoice(this, ID_REMINDERCHOICE, wxDefaultPosition, wxDefaultSize, choices);
    m_reminderCtrl->SetSelection(0);
    gridSizer->Add(m_reminderCtrl, wxSizerFlags().Expand().Align(wxALIGN_CENTER_VERTICAL).Border(wxUP, 5));
    m_reminderEdit = new wxBitmapButton(this, ID_REMINDEREDIT, ICO_EDIT);
    gridSizer->Add(m_reminderEdit, wxSizerFlags().Expand().Align(wxALIGN_CENTER_VERTICAL).Border(wxUP, 5));
    m_reminderEdit->Hide();
    m_mainSizer->Add(gridSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* cancelButton = new wxButton(this, wxID_CANCEL, "", wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(cancelButton);
    btnSizer->Realize();
    m_mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(m_mainSizer);
    m_mainSizer->Fit(this);
    m_mainSizer->SetSizeHints(this);
}

void LaterDialog::OnReminder(wxCommandEvent& event)
{
    int id = event.GetInt();
    switch(id)
    {
        case 0: m_reminder = 5; m_reminderEdit->Hide(); break;
        case 1: m_reminder = 10; m_reminderEdit->Hide(); break;
        case 2: m_reminder = 15; m_reminderEdit->Hide(); break;
        case 3: m_reminder = 30; m_reminderEdit->Hide(); break;
        case 4: m_reminder = 45; m_reminderEdit->Hide(); break;
        case 5: m_reminder = 60; m_reminderEdit->Hide(); break;
        case 6: m_reminder = 120; m_reminderEdit->Hide();break;
        case 7: m_reminder = 1440; m_reminderEdit->Hide(); break;
        default: m_reminder = m_reminderOwn; m_reminderEdit->Show(); break;
    }
    m_mainSizer->Layout();
    m_mainSizer->Fit(this);
}

void LaterDialog::OnReminderEdit(wxCommandEvent &/*event*/)
{
    ReminderDialog dialog(this, m_reminder);
    if(dialog.ShowModal() == wxID_OK)
    {
        m_reminder = dialog.reminder();
        m_reminderOwn = m_reminder;
        wxInt64 mins = m_reminder;
        wxInt64 days = mins/(60*24);
        mins -= days*60*24;
        wxInt64 hours = mins/60;
        mins -= hours*60;
        wxString text = _("Custom");
        if(mins || hours || days)
        {
            text << " (";
        }
        if(days)
        {
            text << wxString::Format(wxPLURAL("%lld day","%lld days",days),days);
            if(mins || hours) text << ", ";
        }
        if(hours)
        {
            text << wxString::Format(wxPLURAL("%lld hour","%lld hours",hours),hours);
            if(mins) text << ", ";
        }
        if(mins)
        {
            text << wxString::Format(wxPLURAL("%lld minute","%lld minutes",mins),mins);
        }
        if(mins || hours || days)
        {
            text << ")";
        }
        m_reminderCtrl->SetString(8, text);
        m_reminderCtrl->SetSelection(8);
        m_mainSizer->Layout();
        m_mainSizer->Fit(this);
    }
}

BEGIN_EVENT_TABLE(EventPanel, wxPanel)
    EVT_BUTTON(ID_BTN_EVENTLATER, EventPanel::OnRemindLater)
    EVT_BUTTON(ID_BTN_EVENTOK, EventPanel::OnOk)
END_EVENT_TABLE()

EventPanel::EventPanel(wxWindow *parent, Event event, EventMsgDialog *dialog)
: wxPanel(parent, wxID_ANY, wxDefaultPosition, wxDefaultSize, wxTAB_TRAVERSAL|wxBORDER_SUNKEN), m_id(event.id()), m_date(event.date()), m_notdelete(event.recurrence()==R_ONCENOTDELETE), m_dialog(dialog)
{
    wxString text = wxEmptyString;
    if(event.reminder() < 0)
    {
        text = _("Now");
    }
    else
    {
        switch(event.reminder())
        {
        case 0: text = _("Now"); break; //0 mins
        case 5: text = _("In 5 Minutes"); break; //5 mins
        case 15: text = _("In 15 Minutes"); break; //15 mins
        case 60: text = _("In 1 Hour"); break; //1 hours
        case 120: text = _("In 2 Hours"); break; //2 hours
        case 1440: text = _("Tomorrow"); break; //1 day
        case 2880: text = _("In 2 Days"); break; //2 days
        default:
        {
            wxInt64 mins = event.reminder();
            wxInt64 days = mins/(60*24);
            mins -= days*60*24;
            wxInt64 hours = mins/60;
            mins -= hours*60;
            if(days == 0)
            {
                if(hours == 0) text = wxString::Format(wxPLURAL("In %lld Minute","In %lld Minutes",mins), mins);
                else
                {
                    if(mins == 0) text = wxString::Format(_("In")+" "+wxPLURAL("%lld Hour","%lld Hours",hours), hours);
                    text = wxString::Format(_("In")+" "+wxPLURAL("%lld Hour","%lld Hours",hours)+", "+wxPLURAL("%lld Minute","%lld Minutes",mins), hours, mins);
                }
            }
            else
            {
                if(hours == 0)
                {
                    if(mins == 0) text = wxString::Format(_("In")+" "+wxPLURAL("%lld Day","%lld Days",days), days);
                    else text = wxString::Format(_("In")+" "+wxPLURAL("%lld Day","%lld Days",days)+", "+wxPLURAL("%lld Minute","%lld Minutes",mins), days, mins);
                }
                else
                {
                    if(mins == 0) text = wxString::Format(_("In")+" "+wxPLURAL("%lld Day","%lld Days",days)+", "+wxPLURAL("%lld Hour","%lld Hours",hours), days, hours);
                    else text = wxString::Format(_("In")+" "+wxPLURAL("%lld Day","%lld Days",days)+", "+wxPLURAL("%lld Hour","%lld Hours",hours)+", "+wxPLURAL("%lld Minute","%lld Minutes",mins), days, hours, mins);
                }
            }
            break;
        }
        }
    }

    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);

    mainSizer->Add(new wxStaticText(this, wxID_ANY, text), 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxFlexGridSizer *gridSizer = new wxFlexGridSizer(2, 5, 5);
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Event:")), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    gridSizer->Add(new wxStaticText(this, wxID_ANY, event.event()), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    if(!event.descr().IsEmpty())
    {
        gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Description:")), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
        gridSizer->Add(new wxStaticText(this, wxID_ANY, event.descr()), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    }
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Date:")), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    gridSizer->Add(new wxStaticText(this, wxID_ANY, dxutils::formatDateTime(event.realEventDate(), dxsettings.dateFormat(), dxsettings.timeFormat(), dxsettings.dateLayout())),
                   wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Recurrence:")), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    gridSizer->Add(new wxStaticText(this, wxID_ANY, event.recurrenceText()), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    if(!event.until().IsSameDate(INFINITY_DATE))
    {
        gridSizer->Add(new wxStaticText(this, wxID_ANY, _("Until:")), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
        gridSizer->Add(new wxStaticText(this, wxID_ANY, dxutils::formatDate(event.until(), dxsettings.dateFormat())), wxSizerFlags().Align(wxALIGN_LEFT | wxALIGN_CENTER_VERTICAL).Border(wxLEFT | wxUP, 5));
    }
    mainSizer->Add(gridSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    wxBoxSizer *btnSizer = new wxBoxSizer(wxHORIZONTAL);
    wxButton* applyButton = new wxButton(this, ID_BTN_EVENTLATER, _("Remind later"));
    btnSizer->Add(applyButton, 0, wxALL, 5);
    wxButton* okButton = new wxButton(this, ID_BTN_EVENTOK, _("OK"));
    btnSizer->Add(okButton, 0, wxALL, 5);
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);
}

wxInt64 EventPanel::id() const
{
    return m_id;
}

wxDateTime EventPanel::date() const
{
    return m_date;
}

void EventPanel::setId(wxInt64 id)
{
    m_id = id;
}

bool EventPanel::notdelete() const
{
    return m_notdelete;
}

void EventPanel::OnRemindLater(wxCommandEvent& /*event*/)
{
    LaterDialog dialog(this);
    if(dialog.ShowModal() == wxID_OK)
    {
        iengine->editEvent(m_id, -(wxDateTime::Now().GetTicks() + dialog.reminder()*60 - m_date.GetTicks())/60);
        m_dialog->removeEvent(m_id);
    }
}

void EventPanel::OnOk(wxCommandEvent &/*event*/)
{
    iengine->setEventReminded(m_id);
    if(iengine->hasRecurrence(m_id)) iengine->recurrentEvent(m_id, false);
    else
    {
        if(!m_notdelete) iengine->removeEvent(m_id, false);
    }
    m_dialog->removeEvent(m_id);
}

IMPLEMENT_DYNAMIC_CLASS(EventMsgDialog, wxDialog)

BEGIN_EVENT_TABLE(EventMsgDialog, wxDialog)
    EVT_BUTTON(wxID_OK, EventMsgDialog::OnOk)
    EVT_BUTTON(wxID_APPLY, EventMsgDialog::OnRemindLater)
    EVT_CLOSE(EventMsgDialog::OnClose)
END_EVENT_TABLE()

EventMsgDialog::EventMsgDialog(wxWindow *parent)
    : wxDialog(parent, wxID_ANY, _("Events")), m_hidden(0)
{
    std::vector<Event> events = iengine->eventsByReminder(wxDateTime::Now());

    wxBoxSizer *mainSizer = new wxBoxSizer(wxVERTICAL);

    m_scrl = new wxScrolledWindow(this);
    wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
    sizer->AddSpacer(3);
    for(size_t i=0; i<events.size(); i++)
    {
        EventPanel *pnl = new EventPanel(m_scrl, events[i], this);
        m_events.push_back(pnl);
        sizer->Add(pnl, 0, wxALL|wxALIGN_CENTER_HORIZONTAL);
        sizer->AddSpacer(3);
    }
    m_scrl->SetSizer(sizer);
    m_scrl->FitInside();
    m_scrl->SetScrollRate(0,5);
    m_scrl->SetMinSize(wxSize(-1,188));
    mainSizer->Add(m_scrl, 1, wxEXPAND, 5);

    wxStdDialogButtonSizer *btnSizer = new wxStdDialogButtonSizer();
    wxButton* okButton = new wxButton(this, wxID_OK, _("OK for All"), wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(okButton);
    wxButton* applyButton = new wxButton(this, wxID_APPLY, _("All remind later"), wxDefaultPosition, wxDefaultSize, 0);
    btnSizer->AddButton(applyButton);
    btnSizer->Realize();
    mainSizer->Add(btnSizer, 0, wxALL|wxALIGN_CENTER_HORIZONTAL, 5);

    this->SetSizer(mainSizer);
    mainSizer->Fit(this);
    mainSizer->SetSizeHints(this);

    okButton->SetFocus();
}

void EventMsgDialog::removeEvent(wxInt64 id)
{
    std::vector<EventPanel*>::iterator it;
    for(it = m_events.begin(); it != m_events.end(); ++it)
    {
        if((*it) && (*it)->id() == id)
        {
            (*it)->setId(-1);
            (*it)->Hide();
            m_hidden++;
            break;
        }
    }
    m_scrl->FitInside();
    if(m_hidden == m_events.size())
    {
        iengine->tooltipPluginForDate();
        EndModal(wxID_OK);
    }
}

void EventMsgDialog::OnOk(wxCommandEvent &/*event*/)
{
    for(size_t i=0; i<m_events.size(); i++)
    {
        if(m_events[i]->id() == -1) continue;
        iengine->setEventReminded(m_events[i]->id());
        if(iengine->hasRecurrence(m_events[i]->id())) iengine->recurrentEvent(m_events[i]->id(), false);
        else
        {
            if(!m_events[i]->notdelete()) iengine->removeEvent(m_events[i]->id(), false);
        }
    }
    iengine->tooltipPluginForDate();
    EndModal(wxID_OK);
}

void EventMsgDialog::OnRemindLater(wxCommandEvent &/*event*/)
{
    LaterDialog dialog(this);
    if(dialog.ShowModal() == wxID_OK)
    {
        for(size_t i=0; i<m_events.size(); i++)
        {
            if(m_events[i]->id() != -1) iengine->editEvent(m_events[i]->id(), -(wxDateTime::Now().GetTicks() + dialog.reminder()*60 - m_events[i]->date().GetTicks())/60);
        }
    }
    iengine->tooltipPluginForDate();
    EndModal(wxID_OK);
}

void EventMsgDialog::OnClose(wxCloseEvent &/*event*/)
{
    EndModal(wxID_CANCEL);
}
