use crate::modules::prelude::*;
use crate::utils::TranslateMetadata;
use crate::modules::types::Type;
use crate::modules::prelude::RawFragment;
use crate::modules::expression::expr::{Expr, ExprType};
use super::fragment::{FragmentKind, FragmentRenderable};
use super::get_variable_name;
use super::var_stmt::VarStmtFragment;

/// Represents a variable expression such as `$var` or `${var}`
#[derive(Debug, Clone, PartialEq, Eq)]
pub enum VarRenderType {
    NameOf,
    BashRef,
    BashValue,
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum VarIndexValue {
    Index(FragmentKind),
    Range(FragmentKind, FragmentKind),
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct VarExprFragment {
    pub name: String,
    // Global id generated by Amber
    pub global_id: Option<usize>,
    // Type of the value
    pub kind: Type,
    // Amber's reference
    pub is_ref: bool,
    // Bash's length getter `${#var}`
    pub is_length: bool,
    // Bash's default value `${var:-default}`
    pub default_value: Option<Box<FragmentKind>>,
    // Quotes around this expression
    pub is_quoted: bool,
    // Bash's `${array[*]}` expansion
    pub is_array_to_string: bool,
    pub render_type: VarRenderType,
    // Amber's array subscript like `arr[0]` or `arr[1..5]`
    pub index: Option<Box<VarIndexValue>>,
}

// Represents variable that resolves to a value. Prefixed with `$`.

impl Default for VarExprFragment {
    fn default() -> Self {
        VarExprFragment {
            name: String::new(),
            global_id: None,
            kind: Type::Generic,
            is_ref: false,
            is_length: false,
            is_array_to_string: false,
            is_quoted: true,
            render_type: VarRenderType::BashValue,
            index: None,
            default_value: None,
        }
    }
}

impl VarExprFragment {
    pub fn new(name: &str, kind: Type) -> Self {
        VarExprFragment {
            name: name.to_string(),
            kind,
            ..Default::default()
        }
    }

    pub fn with_global_id<T: Into<Option<usize>>>(mut self, id: T) -> Self {
        self.global_id = id.into();
        self
    }

    pub fn with_ref(mut self, is_ref: bool) -> Self {
        self.is_ref = is_ref;
        self
    }

    pub fn with_array_to_string(mut self, is_array_to_string: bool) -> Self {
        self.is_array_to_string = is_array_to_string;
        self
    }

    pub fn from_stmt(stmt: &VarStmtFragment) -> Self {
        VarExprFragment {
            name: stmt.name.clone(),
            global_id: stmt.global_id,
            kind: stmt.kind.clone(),
            is_ref: stmt.is_ref,
            ..Default::default()
        }
    }

    pub fn with_index_by_expr<T: Into<Option<Expr>>>(mut self, meta: &mut TranslateMetadata, index: T) -> Self {
        if let Some(index) = index.into() {
            let index = match index.value {
                Some(ExprType::Range(range)) => {
                    let (offset, length) = range.get_array_index(meta);
                    VarIndexValue::Range(offset, length)
                }
                Some(ExprType::Neg(neg)) => {
                    let index = neg.get_array_index(meta);
                    VarIndexValue::Index(index)
                }
                _ => {
                    let index = index.translate_eval(meta, true);
                    VarIndexValue::Index(index)
                }
            };
            self.index = Some(Box::new(index));
        }
        self
    }

    pub fn with_index_by_value<T: Into<Option<VarIndexValue>>>(mut self, index: T) -> Self {
        self.index = index.into().map(Box::new);
        self
    }

    pub fn with_default_value<T: Into<Option<FragmentKind>>>(mut self, default_value: T) -> Self {
        self.default_value = default_value.into().map(Box::new);
        self
    }

    pub fn with_length_getter(mut self, value: bool) -> Self {
        self.is_length = value;
        self
    }

    pub fn with_render_type(mut self, render_type: VarRenderType) -> Self {
        self.render_type = render_type;
        self
    }

    pub fn with_quotes(mut self, is_quoted: bool) -> Self {
        self.is_quoted = is_quoted;
        self
    }

    pub fn get_name(&self) -> String {
        get_variable_name(&self.name, self.global_id)
    }

    // Returns the variable name in the bash context Ex. "varname"
    pub fn render_bash_reference(self, meta: &mut TranslateMetadata) -> String {
        let dollar = meta.gen_dollar();
        let mut name = self.get_name();
        // Dereference variable if it's a reference and is passed by reference
        if self.is_ref {
            name = format!("{dollar}{name}");
        }

        if self.is_quoted {
            let quote = meta.gen_quote();
            format!("{quote}{name}{quote}")
        } else {
            name
        }
    }

    // Returns the variable value in the bash context Ex. "$varname" or "${varname[@]}"
    pub fn render_bash_value(mut self, meta: &mut TranslateMetadata) -> String {
        let name = self.get_name();
        let index = self.index.take();
        let default_value = self.default_value.take();
        let prefix = self.get_variable_prefix();
        let suffix = self.get_variable_suffix(meta, index, default_value);

        if self.is_ref {
            self.render_deref_variable(meta, prefix, &name, &suffix)
        } else {
            let quote = if self.is_quoted { meta.gen_quote() } else { "" };
            let dollar = meta.gen_dollar();
            format!("{quote}{dollar}{{{prefix}{name}{suffix}}}{quote}")
        }
    }

    // Get variable prefix ${PREFIX-varname-suffix}
    fn get_variable_prefix(&self) -> &'static str {
        if self.is_length {
            "#"
        } else {
            ""
        }
    }

    // Get variable suffix ${prefix-varname-SUFFIX}
    fn get_variable_suffix(
        &self,
        meta: &mut TranslateMetadata,
        index: Option<Box<VarIndexValue>>,
        default_value: Option<Box<FragmentKind>>
    ) -> String {
        let default_value = default_value
            .map(|value| value.to_string(meta))
            .map(|value| format!(":-{value}"))
            .unwrap_or_default();
        match (&self.kind, index.map(|var| *var)) {
            (Type::Array(_), Some(VarIndexValue::Range(offset, length))) => {
                if self.default_value.is_some() {
                    panic!("It's impossible to render default value when slicing");
                }
                let offset = offset.with_quotes(false).to_string(meta);
                let length = length.with_quotes(false).to_string(meta);
                format!("[@]:{offset}:{length}")
            }
            (_, Some(VarIndexValue::Index(index))) => {
                let index = index.with_quotes(false).to_string(meta);
                format!("[{index}]{default_value}")
            }
            (Type::Array(_), None) if self.is_array_to_string => {
                format!("[*]{default_value}")
            }
            (Type::Array(_), None) => {
                format!("[@]{default_value}")
            }
            _ => {
                default_value
            }
        }
    }

    fn render_deref_variable(self, meta: &mut TranslateMetadata, prefix: &str, name: &str, suffix: &str) -> String {
        let arr_open = if self.kind.is_array() { "(" } else { "" };
        let arr_close = if self.kind.is_array() { ")" } else { "" };
        let quote = if self.is_quoted { meta.gen_quote() } else { "" };
        let dollar = meta.gen_dollar();
        if prefix.is_empty() && suffix.is_empty() {
            return format!("{quote}{dollar}{{!{name}}}{quote}");
        }
        let id = meta.gen_value_id();
        let eval_value = format!("{prefix}${{{name}}}{suffix}");
        let var_name = format!("{name}_deref_{id}");
        meta.stmt_queue.push_back(RawFragment::from(
            format!("eval \"local {var_name}={arr_open}\\\"\\${{{eval_value}}}\\\"{arr_close}\"")
        ).to_frag());

        if self.kind.is_array() {
            format!("{quote}{dollar}{{{var_name}[@]}}{quote}")
        } else {
            format!("{quote}{dollar}{{{var_name}}}{quote}")
        }
    }
}

impl FragmentRenderable for VarExprFragment {
    fn to_string(self, meta: &mut TranslateMetadata) -> String {
        match self.render_type {
            VarRenderType::NameOf => self.get_name(),
            VarRenderType::BashRef => self.render_bash_reference(meta),
            VarRenderType::BashValue => self.render_bash_value(meta),
        }
    }

    fn to_frag(self) -> FragmentKind {
        FragmentKind::VarExpr(self)
    }
}
